<?php

namespace WPRemoteMediaExt\Guzzle\Tests\Service\Command;

use WPRemoteMediaExt\Guzzle\Http\Message\EntityEnclosingRequest;
use WPRemoteMediaExt\Guzzle\Http\Message\Response;
use WPRemoteMediaExt\Guzzle\Service\Client;
use WPRemoteMediaExt\Guzzle\Service\Command\OperationCommand;
use WPRemoteMediaExt\Guzzle\Service\Description\Operation;
use WPRemoteMediaExt\Guzzle\Service\Description\ServiceDescription;
use WPRemoteMediaExt\Guzzle\Service\Command\DefaultRequestSerializer;
use WPRemoteMediaExt\Guzzle\Service\Resource\Model;
use WPRemoteMediaExt\Guzzle\Service\Command\LocationVisitor\VisitorFlyweight;

/**
 * @covers WPRemoteMediaExt\Guzzle\Service\Command\OperationCommand
 */
class OperationCommandTest extends \Guzzle\Tests\GuzzleTestCase
{
    public function testHasRequestSerializer()
    {
        $operation = new OperationCommand();
        $a = $operation->getRequestSerializer();
        $b = new DefaultRequestSerializer(VisitorFlyweight::getInstance());
        $operation->setRequestSerializer($b);
        $this->assertNotSame($a, $operation->getRequestSerializer());
    }

    public function testPreparesRequestUsingSerializer()
    {
        $op = new OperationCommand(array(), new Operation());
        $op->setClient(new Client());
        $s = $this->getMockBuilder('WPRemoteMediaExt\Guzzle\Service\Command\RequestSerializerInterface')
            ->setMethods(array('prepare'))
            ->getMockForAbstractClass();
        $s->expects($this->once())
            ->method('prepare')
            ->will($this->returnValue(new EntityEnclosingRequest('POST', 'http://foo.com')));
        $op->setRequestSerializer($s);
        $op->prepare();
    }

    public function testParsesResponsesWithResponseParser()
    {
        $op = new OperationCommand(array(), new Operation());
        $p = $this->getMockBuilder('WPRemoteMediaExt\Guzzle\Service\Command\ResponseParserInterface')
            ->setMethods(array('parse'))
            ->getMockForAbstractClass();
        $p->expects($this->once())
            ->method('parse')
            ->will($this->returnValue(array('foo' => 'bar')));
        $op->setResponseParser($p);
        $op->setClient(new Client());
        $request = $op->prepare();
        $request->setResponse(new Response(200), true);
        $this->assertEquals(array('foo' => 'bar'), $op->execute());
    }

    public function testParsesResponsesUsingModelParserWhenMatchingModelIsFound()
    {
        $description = ServiceDescription::factory(array(
            'operations' => array(
                'foo' => array('responseClass' => 'bar', 'responseType' => 'model')
            ),
            'models' => array(
                'bar' => array(
                    'type' => 'object',
                    'properties' => array(
                        'Baz' => array('type' => 'string', 'location' => 'xml')
                    )
                )
            )
        ));

        $op = new OperationCommand(array(), $description->getOperation('foo'));
        $op->setClient(new Client());
        $request = $op->prepare();
        $request->setResponse(new Response(200, array(
            'Content-Type' => 'application/xml'
        ), '<Foo><Baz>Bar</Baz></Foo>'), true);
        $result = $op->execute();
        $this->assertEquals(new Model(array('Baz' => 'Bar')), $result);
    }

    public function testAllowsRawResponses()
    {
        $description = new ServiceDescription(array(
            'operations' => array('foo' => array('responseClass' => 'bar', 'responseType' => 'model')),
            'models'     => array('bar' => array())
        ));
        $op = new OperationCommand(array(
            OperationCommand::RESPONSE_PROCESSING => OperationCommand::TYPE_RAW
        ), $description->getOperation('foo'));
        $op->setClient(new Client());
        $request = $op->prepare();
        $response = new Response(200, array(
            'Content-Type' => 'application/xml'
        ), '<Foo><Baz>Bar</Baz></Foo>');
        $request->setResponse($response, true);
        $this->assertSame($response, $op->execute());
    }
}
